/**
 * @file    iOSDependencyInfo.cs
 * 
 * @author  jschoi
 * @date    2024~
 * @copyright Copyright © Com2uS Platform Corporation. All Right Reserved.
 * @defgroup Hive.Unity.Editor
 * @{
 * @brief HIVE External Dependency Info <br/><br/>
 */

using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEditor;
using System;
using System.IO;
using System.ComponentModel;
using System.Reflection;
using System.Xml;

namespace Hive.Unity.Editor
{    
    using ExternalDependencyDictionary = Dictionary<ExternalDependencyType, bool>;

    public class iOSDependencyInfo {

        // iOS 플랫폼 버전 파일 구조화
        [System.Serializable]
        public class VersionFileInfo
        {
            public string type;
            public string os;
            public string version;
        }

        // iOS 플랫폼 버전 정보 획득
        // - /Assets/Hive_SDK_v4/Version 에 있는 HiveSDKPlatformiOSVersion.json 파일에서 "version" 키의 값을 반환합니다.
        public static string getiOSDependencyVersion()
        {
            try
            {
                var json = System.IO.File.ReadAllText(Path.Combine(Application.dataPath, "Hive_SDK_v4/Version/HiveSDKPlatformiOSVersion.json"));
                var versionFileInfo = JsonUtility.FromJson<VersionFileInfo>(json);
                return versionFileInfo.version;
            }
            catch (Exception e)
            {
                return "";
            }  
        }

        // <iOSPods> 태그 이하 모든 정보를 수집해서 반환
        private static HashSet<string> getiOSPodsSet(ExternalDependencyDictionary dictionary, iOSDependencyDataType dataType) {
            HashSet<string> set = new HashSet<string>();    // HashSet 중복 허용하지 않음

            // dictionary에서 true 설정된 provider 정보 획득
            foreach(KeyValuePair<ExternalDependencyType, bool> pair in dictionary) {
                if(pair.Value == true)  {

                    iOSDependencyXmlInfo dependencyXmlInfo = getDependencyInfo(pair.Key);
                    if(dependencyXmlInfo != null)  {

                        switch(dataType) {
                            case iOSDependencyDataType.IosSource:
                                foreach(string data in dependencyXmlInfo.iosSource) {
                                    set.Add(data);
                                }
                                break;
                            case iOSDependencyDataType.IosPod:
                                foreach(string data in dependencyXmlInfo.iosPod) {
                                    set.Add(data);
                                }
                                break;
                        }
                    }
                }
            }

            // HiveSDK default 정보 획득(ExternalDependecyType에 속하지 않아 항상 포함되도록)
            HiveDefault hiveDefault = new HiveDefault(ExternalDependencyUtil.isSampleBuild());

            switch(dataType) {
                case iOSDependencyDataType.IosSource:
                    foreach(string data in hiveDefault.iosSource) {
                        set.Add(data);
                    }
                    break;
                case iOSDependencyDataType.IosPod:
                    foreach(string data in hiveDefault.iosPod) {
                        set.Add(data);
                    }
                    break;
            }

            return set;
        }

        public static string getiOSPodsData(ExternalDependencyDictionary dictionary) {
            HashSet<string> iosSource = new HashSet<string>();
            HashSet<string> iosPod = new HashSet<string>();

            iosSource = getiOSPodsSet(dictionary, iOSDependencyDataType.IosSource);
            iosPod = getiOSPodsSet(dictionary, iOSDependencyDataType.IosPod);

            string xmlText = makeUpXmlString(iosSource, iosPod);

            return xmlText;
        }

        private static string makeUpXmlString(HashSet<string> iosSource, HashSet<string> iosPod) {
            string space = "    ";
            string xmlText = "";

            // iosPods
            xmlText += space + "<iosPods>\n";
            xmlText += space + space + "<sources>\n";

            foreach(string data in iosSource) {
                if(data != null && !data.Equals("")) {
                    xmlText += space + space + space;
                    xmlText += "<source>" + data + "</source>\n";
                }
            }
            xmlText += space + space + "</sources>\n";

            // iosPod
            foreach(string data in iosPod) {
                if(data != null && !data.Equals("")) {
                    xmlText += space + space;
                    xmlText += "<iosPod " + data + "/>\n";
                }
            }
            xmlText += space + "</iosPods>\n";

            return xmlText;
        }

        // DependencyInfo를 상속받은 클래스 중 type에 해당하는 클래스 반환
        private static iOSDependencyXmlInfo getDependencyInfo(ExternalDependencyType type) {
            switch(type) {
                case ExternalDependencyType.Hercules:
                    return new Hercules(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.AnalyticsConsentMode:
                    return new AnalyticsConsentMode(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MatchMaking:
                    return new HiveMatchMaking(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Chat:
                    return new HiveChat(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.DataStore:
                    return new HiveDataStore(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.HiveDeviceManagement:
                    return new HiveDeviceManagement(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.GoogleInAppReview:
                    return new HiveGoogleInAppReview(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.GoogleInAppUpdate:
                    return new HiveGoogleInAppUpdate(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.HivePromotion:
                    return new HivePromotion(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.HiveRealNameVerification:
                    return new HiveRealNameVerification(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MarketAmazon:
                    return new MarketAmazon(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MarketApple:
                    return new MarketApple(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MarketGoogle:
                    return new MarketGoogle(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MarketHiveStore:
                    return new MarketHiveStore(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MarketHuawei:
                    return new MarketHuawei(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MarketLebi:
                    return new MarketLebi(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MarketOneStore:
                    return new MarketOneStore(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MarketSamsung:
                    return new MarketSamsung(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Adjust:
                    return new ProviderAdjust(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Airbridge:
                    return new ProviderAirbridge(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Apple:
                    return new ProviderApple(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.AppleSignIn:
                    return new ProviderAppleSignIn(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.AppsFlyer:
                    return new ProviderAppsFlyer(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Facebook:
                    return new ProviderFacebook(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Firebase:
                    return new ProviderFirebase(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Google:
                    return new ProviderGoogle(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.GooglePlayGames:
                    return new ProviderGooglePlayGames(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.HiveMembership:
                    return new ProviderHiveMembership(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Huawei:
                    return new ProviderHuawei(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Line:
                    return new ProviderLine(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.QQ:
                    return new ProviderQQ(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Recaptcha:
                    return new ProviderRecaptcha(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Singular:
                    return new ProviderSingular(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.VK:
                    return new ProviderVK(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Wechat:
                    return new ProviderWechat(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Weverse:
                    return new ProviderWeverse(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.X:
                    return new ProviderX(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Steam:
                    return new ProviderSteam(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Telegram:
                    return new ProviderTelegram(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.PushAmazonAdm:
                    return new PushAmazonAdm(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.PushGoogleFcm:
                    return new PushGoogleFcm(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.HiveAdultConfirm:
                    return new  HiveAdultConfirm(ExternalDependencyUtil.isSampleBuild());
                default:
                    return null;
            }
        }

        public abstract class iOSDependencyXmlInfo {
            public List<string> iosSource;
            public List<string> iosPod;
        }

        public enum iOSDependencyDataType {
            IosSource,
            IosPod
        }

        public class iOSExternalLibraryVersion {
            public class HiveModule {
                public static readonly string sdWebImage = "5.21.1";
                public static readonly string hercules = "2025.02.18";
                public static readonly string recaptcha = "18.7.0";
            }
            public class Auth {
                public static readonly string facebook = "18.0.0";
                public static readonly string googleSignIn = "9.0.0";
                public static readonly string line = "5.13.0";
                public static readonly string wechatOpenSdk = "2.0.4";
            }
            public class Push {
                //
            }
            public class Market {
                //
            }
            public class Analytics {
                public static readonly string adjust = "5.4.3";
                public static readonly string airbridge = "4.7.0";
                public static readonly string appsflyer = "6.17.2";
                public static readonly string firebase = "11.15.0";
                public static readonly string singular = "12.8.1";
            }
            public class Etc {
                //
            }
        }

        // HiveDefault : ExternalDependencyType에 없지만 기본 포함 필요
        // (구 HIVEDefsult, 구 HIVESDKDependencies)
        class HiveDefault : iOSDependencyXmlInfo {
            public HiveDefault() {
                iosSource = new List<string>() {"https://github.com/Com2uSPlatformCorp/HiveSDK-iOS.git"};
                iosPod = new List<string>() {$"name=\"HiveExtensions\" version=\"{getiOSDependencyVersion()}\"",
                                            $"name=\"HiveSDKUnityPlugin\" version=\"{getiOSDependencyVersion()}\""};
            }
            public HiveDefault(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {$"name=\"SDWebImage\" version=\"{iOSExternalLibraryVersion.HiveModule.sdWebImage}\" bitcodeEnabled=\"false\" minTargetSdk=\"12.0\""};
                }
            }
        }

        class Hercules : iOSDependencyXmlInfo {
            public Hercules() {
                iosSource = new List<string>() {"https://github.com/Com2uSPlatformCorp/Hercules-iOS.git"};
                iosPod = new List<string>() {"name=\"HerculesSwift\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public Hercules(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {"https://github.com/Com2uSPlatformCorp/Hercules-iOS.git"};
                    iosPod = new List<string>() {$"name=\"Hercules\" version=\"{iOSExternalLibraryVersion.HiveModule.hercules}\" bitcodeEnabled=\"false\" minTargetSdk=\"12.0\""};
                }
            }
        }
        
        class AnalyticsConsentMode : iOSDependencyXmlInfo {
            public AnalyticsConsentMode() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAnalyticsConsentMode\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public AnalyticsConsentMode(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class HiveDataStore : iOSDependencyXmlInfo {
            public HiveDataStore() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveDataStore\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public HiveDataStore(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class HiveDeviceManagement : iOSDependencyXmlInfo {
            public HiveDeviceManagement() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveDeviceManagement\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public HiveDeviceManagement(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class HiveGoogleInAppReview : iOSDependencyXmlInfo {
            public HiveGoogleInAppReview() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {""};
            }
            public HiveGoogleInAppReview(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class HiveGoogleInAppUpdate : iOSDependencyXmlInfo {
            public HiveGoogleInAppUpdate() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {""};
            }
            public HiveGoogleInAppUpdate(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class HiveMatchMaking : iOSDependencyXmlInfo {
            public HiveMatchMaking() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveMatchMaking\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public HiveMatchMaking(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class HiveChat : iOSDependencyXmlInfo {
            public HiveChat() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {{"name=\"HiveChat\" version=\"" + getiOSDependencyVersion() + "\""}};
            }
            public HiveChat(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class HivePromotion : iOSDependencyXmlInfo {
            public HivePromotion() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HivePromotion\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public HivePromotion(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class HiveRealNameVerification : iOSDependencyXmlInfo {
            public HiveRealNameVerification() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveRealNameVerification\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public HiveRealNameVerification(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }
        
        class HiveAdultConfirm : iOSDependencyXmlInfo {
            public HiveAdultConfirm() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAdultConfirm\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public HiveAdultConfirm(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class MarketAmazon : iOSDependencyXmlInfo {
            public MarketAmazon() {
                iosSource = new List<string>() { "" };
                iosPod = new List<string>() { "" };
            }
            public MarketAmazon(bool sampleBuild) : this() {
                if (sampleBuild) {
                    iosSource = new List<string>() { "" };
                    iosPod = new List<string>() { "" };
                }
            }
        }

        class MarketApple : iOSDependencyXmlInfo {
            public MarketApple() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveIAPV4\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public MarketApple(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class MarketGoogle : iOSDependencyXmlInfo {
            public MarketGoogle() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {""};
            }
            public MarketGoogle(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class MarketHiveStore : iOSDependencyXmlInfo {
            public MarketHiveStore() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {""};
            }
            public MarketHiveStore(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class MarketHuawei : iOSDependencyXmlInfo {
            public MarketHuawei() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {""};
            }
            public MarketHuawei(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class MarketLebi : iOSDependencyXmlInfo {
            public MarketLebi() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {""};
            }
            public MarketLebi(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class MarketOneStore : iOSDependencyXmlInfo {
            public MarketOneStore() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {""};
            }
            public MarketOneStore(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class MarketSamsung : iOSDependencyXmlInfo {
            public MarketSamsung() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {""};
            }
            public MarketSamsung(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class ProviderAdjust : iOSDependencyXmlInfo {
            public ProviderAdjust() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAnalyticsProviderAdjust\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderAdjust(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {"https://github.com/CocoaPods/Specs.git"};
                    iosPod = new List<string>() {$"name=\"Adjust\" version=\"{iOSExternalLibraryVersion.Analytics.adjust}\" bitcodeEnabled=\"false\" minTargetSdk=\"12.0\""};
                }
            }
        }

        class ProviderAirbridge : iOSDependencyXmlInfo {
            public ProviderAirbridge() {
                iosSource = new List<string>() { "" };
                iosPod = new List<string>() { "name=\"HiveAnalyticsProviderAirbridge\" version=\"" + getiOSDependencyVersion() + "\"" };
            }
            public ProviderAirbridge(bool sampleBuild) : this() {
                if (sampleBuild) {
                    iosSource = new List<string>() { "https://github.com/CocoaPods/Specs.git" };
                    iosPod = new List<string>() { $"name=\"airbridge-ios-sdk\" version=\"{iOSExternalLibraryVersion.Analytics.airbridge}\" bitcodeEnabled=\"false\" minTargetSdk=\"12.0\"" };
                }
            }
        }

        class ProviderApple : iOSDependencyXmlInfo {
            public ProviderApple() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAuthV4ProviderAppleGameCenter\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderApple(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class ProviderAppleSignIn : iOSDependencyXmlInfo {
            public ProviderAppleSignIn() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAuthV4ProviderAppleSignIn\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderAppleSignIn(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class ProviderAppsFlyer : iOSDependencyXmlInfo {
            public ProviderAppsFlyer() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAnalyticsProviderAppsFlyer\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderAppsFlyer(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {$"name=\"AppsFlyerFramework\" version=\"{iOSExternalLibraryVersion.Analytics.appsflyer}\" bitcodeEnabled=\"false\" minTargetSdk=\"9.0\""};
                }
            }
        }

        class ProviderFacebook : iOSDependencyXmlInfo {
            public ProviderFacebook() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAuthV4ProviderFacebook\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderFacebook(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {$"name=\"FBSDKCoreKit\" version=\"{iOSExternalLibraryVersion.Auth.facebook}\" bitcodeEnabled=\"false\" minTargetSdk=\"12.0\"",
                                                $"name=\"FBSDKLoginKit\" version=\"{iOSExternalLibraryVersion.Auth.facebook}\" bitcodeEnabled=\"false\" minTargetSdk=\"12.0\"",
                                                $"name=\"FBSDKShareKit\" version=\"{iOSExternalLibraryVersion.Auth.facebook}\" bitcodeEnabled=\"false\" minTargetSdk=\"12.0\"",
                                                $"name=\"FBSDKGamingServicesKit\" version=\"{iOSExternalLibraryVersion.Auth.facebook}\" bitcodeEnabled=\"false\" minTargetSdk=\"12.0\""};
                }
            }
        }

        class ProviderFirebase : iOSDependencyXmlInfo {
            public ProviderFirebase() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAnalyticsProviderFirebase\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderFirebase(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {$"name=\"FirebaseCore\" version=\"{iOSExternalLibraryVersion.Analytics.firebase}\" bitcodeEnabled=\"false\" minTargetSdk=\"10.0\"",
                                                $"name=\"FirebaseAnalytics\" version=\"{iOSExternalLibraryVersion.Analytics.firebase}\" bitcodeEnabled=\"false\" minTargetSdk=\"10.0\""};
                }
            }
        }

        class ProviderGoogle : iOSDependencyXmlInfo {
            public ProviderGoogle() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAuthV4ProviderGoogle\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderGoogle(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {$"name=\"GoogleSignIn\" version=\"{iOSExternalLibraryVersion.Auth.googleSignIn}\" bitcodeEnabled=\"false\" minTargetSdk=\"12.0\""};
                }
            }
        }

        class ProviderGooglePlayGames : iOSDependencyXmlInfo {
            public ProviderGooglePlayGames() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {""};
            }
            public ProviderGooglePlayGames(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class ProviderHiveMembership : iOSDependencyXmlInfo {
            public ProviderHiveMembership() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAuthV4ProviderHive\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderHiveMembership(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class ProviderHuawei : iOSDependencyXmlInfo {
            public ProviderHuawei() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {""};
            }
            public ProviderHuawei(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class ProviderLine : iOSDependencyXmlInfo {
            public ProviderLine() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAuthV4ProviderLine\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderLine(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {$"name=\"LineSDKSwift\" version=\"{iOSExternalLibraryVersion.Auth.line}\" bitcodeEnabled=\"false\" minTargetSdk=\"11.0\""};
                }
            }
        }

        class ProviderQQ : iOSDependencyXmlInfo {
            public ProviderQQ() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAuthV4ProviderQQ\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderQQ(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class ProviderRecaptcha : iOSDependencyXmlInfo {
            public ProviderRecaptcha() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveGoogleRecaptcha\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderRecaptcha(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {$"name=\"RecaptchaEnterprise\" version=\"{iOSExternalLibraryVersion.HiveModule.recaptcha}\" bitcodeEnabled=\"false\" minTargetSdk=\"11.0\""};
                }
            }
        }

        class ProviderSingular : iOSDependencyXmlInfo {
            public ProviderSingular() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAnalyticsProviderSingular\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderSingular(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {$"name=\"Singular-SDK\" version=\"{iOSExternalLibraryVersion.Analytics.singular}\" bitcodeEnabled=\"false\" minTargetSdk=\"8.0\""};
                }
            }
        }

        class ProviderVK : iOSDependencyXmlInfo {
            public ProviderVK() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAuthV4ProviderVK\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderVK(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class ProviderWechat : iOSDependencyXmlInfo {
            public ProviderWechat() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAuthV4ProviderWechat\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderWechat(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {$"name=\"WechatOpenSDK-XCFramework\" version=\"{iOSExternalLibraryVersion.Auth.wechatOpenSdk}\" bitcodeEnabled=\"false\" minTargetSdk=\"12.0\""};
                }
            }
        }

        class ProviderWeverse : iOSDependencyXmlInfo {
            public ProviderWeverse() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAuthV4ProviderWeverse\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderWeverse(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class ProviderSteam : iOSDependencyXmlInfo {
            public ProviderSteam() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAuthV4ProviderSteam\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderSteam(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class ProviderX : iOSDependencyXmlInfo {
            public ProviderX() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAuthV4ProviderX\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderX(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class ProviderTelegram : iOSDependencyXmlInfo {
            public ProviderTelegram() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {"name=\"HiveAuthV4ProviderTelegram\" version=\"" + getiOSDependencyVersion() + "\""};
            }
            public ProviderTelegram(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class PushAmazonAdm : iOSDependencyXmlInfo {
            public PushAmazonAdm() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {""};
            }
            public PushAmazonAdm(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }

        class PushGoogleFcm : iOSDependencyXmlInfo {
            public PushGoogleFcm() {
                iosSource = new List<string>() {""};
                iosPod = new List<string>() {""};
            }
            public PushGoogleFcm(bool sampleBuild) : this() {
                if(sampleBuild) {
                    iosSource = new List<string>() {""};
                    iosPod = new List<string>() {""};
                }
            }
        }
    }


}
